<?php

namespace App\Traits;

use Carbon\CarbonPeriod;
use Illuminate\Database\Eloquent\Casts\Attribute;

trait HasTiming
{
    public function lastTimeToJoin(): Attribute
    {
        return Attribute::make(
            get : function () {
                return $this->start_at->subDays(1)->endOfDay();
            }
        );
    }

    public function scopeOnline($query)
    {
        return $query->where('is_online', '1');
    }

    public function scopeOffline($query)
    {
        return $query->where('is_online', '0');
    }

    public function scopeStarted($query)
    {
        return $query->where('start_at', '<=', now());
    }

    public function scopeNotStarted($query)
    {
        return $query->where('start_at', '>', now());
    }

    public function scopeBeforeStartByDays($query, $days)
    {
        return $query->where('start_at', '<=', now()->addDays($days));
    }

    public function scopeEnded($query)
    {
        return $query->where('end_at', '<=', now());
    }

    public function scopeNotEnded($query)
    {
        return $query->where('end_at', '>', now());
    }

    public function scopeStartThisWeek($query)
    {
        return $query->whereBetween('start_at', [now()->startOfWeek(), now()->endOfWeek()]);
    }

    public function isEnded()
    {
        return $this->end_at->isPast();
    }

    public function isStarted()
    {
        return $this->start_at->isPast();
    }

    public function isNotStarted()
    {
        return $this->start_at->isFuture();
    }

    public function isNotEnded()
    {
        return $this->end_at->isFuture();
    }

    public function isOnline()
    {
        return $this->is_online;
    }

    public function isOffline()
    {
        return ! $this->is_online;
    }

    public function stillCanJoinOrLeave(): bool
    {
        $last_time_to_join = $this->last_time_to_join;

        if ($last_time_to_join->isPast()) {
            return false;
        }

        return true;
    }

    public function getDays()
    {
        return CarbonPeriod::create($this->start_at->startOfDay(), $this->end_at->endOfDay());
    }

    public function getDaysCount()
    {
        return $this->getDays()->count();
    }

    public function timingStatus(): Attribute
    {
        return Attribute::make(
            get : function () {
                if ($this->isEnded()) {
                    return 'ended';
                }

                if ($this->start_at->isToday()) {
                    return 'today';
                }

                if ($this->start_at->isTomorrow()) {
                    return 'tomorrow';
                }

                if ($this->start_at->isPast()) {
                    return 'started';
                }

                return 'not_started';
            }
        );
    }
}
