<?php

namespace App\Models;

use App\Enums\Gender;
use App\Traits\HasTiming;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Database\Eloquent\SoftDeletes;

class Course extends Model
{
    use HasFactory;
    use HasTiming;
    use SoftDeletes;

    protected $casts = [
        'start_at' => 'datetime',
        'end_at' => 'datetime',
        'is_for_girls' => 'boolean',
        'is_for_boys' => 'boolean',
        'is_online' => 'boolean',
        'is_for_edu_job' => 'boolean',
        'is_for_manager_job' => 'boolean',
        'is_for_all_offices' => 'boolean',
    ];

    public function scopeAllowedCourseForTrainee(Builder $builder, User $user): Builder
    {
        $has_edu_job = $user->has_edu_job;
        $has_manager_job = $user->has_manager_job;
        $office_id = $user->office_id;
        $is_boy = $user->gender == Gender::MALE;
        $is_girl = $user->gender == Gender::FEMALE;

        return $builder->where(function ($query) use ($has_edu_job, $has_manager_job, $is_boy, $is_girl) {
            $query->when($is_boy, function ($query) use ($has_edu_job, $has_manager_job) {
                $query->where('is_for_boys', true)
                    ->when($has_edu_job, function ($query) {
                        $query->orWhere('is_for_edu_job', true);
                    })
                    ->when($has_manager_job, function ($query) {
                        $query->orWhere('is_for_manager_job', true);
                    });
            })->when($is_girl, function ($query) use ($has_edu_job, $has_manager_job) {
                $query->where('is_for_girls', true)
                    ->when($has_edu_job, function ($query) {
                        $query->orWhere('is_for_edu_job', true);
                    })
                    ->when($has_manager_job, function ($query) {
                        $query->orWhere('is_for_manager_job', true);
                    });
            });
        })->where(function ($query) use ($office_id) {
            $query->where('is_for_all_offices', true)
                ->orWhereHas('offices', function ($sub_query) use ($office_id) {
                    $sub_query->where('office_id', $office_id);
                });
        });
    }

    public function scopeAllowedCourseForCurrentTrainee(Builder $builder): Builder
    {
        $user = auth()->user();
        $has_edu_job = $user->has_edu_job;
        $has_manager_job = $user->has_manager_job;
        $office_id = $user->office_id;
        $is_boy = $user->gender == Gender::MALE;
        $is_girl = $user->gender == Gender::FEMALE;

        return $builder->where(function ($query) use ($has_edu_job, $has_manager_job, $is_boy, $is_girl) {
            $query->when($is_boy, function ($query) use ($has_edu_job, $has_manager_job) {
                $query->where('is_for_boys', true)
                    ->when($has_edu_job, function ($query) {
                        $query->orWhere('is_for_edu_job', true);
                    })
                    ->when($has_manager_job, function ($query) {
                        $query->orWhere('is_for_manager_job', true);
                    });
            })->when($is_girl, function ($query) use ($has_edu_job, $has_manager_job) {
                $query->where('is_for_girls', true)
                    ->when($has_edu_job, function ($query) {
                        $query->orWhere('is_for_edu_job', true);
                    })
                    ->when($has_manager_job, function ($query) {
                        $query->orWhere('is_for_manager_job', true);
                    });
            });
        })->where(function ($query) use ($office_id) {
            $query->where('is_for_all_offices', true)
                ->orWhereHas('offices', function ($sub_query) use ($office_id) {
                    $sub_query->where('office_id', $office_id);
                });
        });
    }

    public function scopeAllowedCoursesForCurrentTrainer(Builder $builder): Builder
    {
        return $builder->whereHas('trainers', function ($query) {
            $query->where('trainer_id', auth()->id());
        });
    }

    public function remainingTraineesCount(): Attribute
    {
        return Attribute::make(
            get : function () {
                return $this->max_trainees - $this->trainees()->count();
            }
        );
    }

    public function scopeOfCurrentTrainee(Builder $builder): Builder
    {
        return $builder->whereHas('trainees', function ($query) {
            $query->where('trainee_id', auth()->id());
        });
    }

    public function scopeOfCurrentTrainer(Builder $builder): Builder
    {
        return $builder->whereHas('trainers', function ($query) {
            $query->where('trainer_id', auth()->id());
        });
    }

    public function type(): BelongsTo
    {
        return $this->belongsTo(CourseType::class, 'course_type_id');
    }

    public function field(): BelongsTo
    {
        return $this->belongsTo(CourseField::class, 'course_field_id');
    }

    public function trainers(): BelongsToMany
    {
        return $this->belongsToMany(
            User::class,
            (new CourseTrainer())->getTable(),
            'course_id',
            'trainer_id'
        )
            ->where('is_trainer', true)
            ->withPivot('id')
            ->withTimestamps();
    }

    public function trainees(): BelongsToMany
    {
        return $this->belongsToMany(
            User::class,
            (new CourseTrainee())->getTable(),
            'course_id',
            'trainee_id'
        )
            ->where('is_trainee', true)
            ->withPivot('id')
            ->withTimestamps();
    }

    public function attendances(): HasManyThrough
    {
        return $this->hasManyThrough(
            Attendance::class,
            CourseTrainee::class,
            'course_id',
            'course_trainee_id',
            'id',
            'id'
        );
    }

    public function offices(): BelongsToMany
    {
        return $this->belongsToMany(
            Office::class,
            (new CourseOffice())->getTable(),
            'course_id',
            'office_id'
        )->withPivot('id');
    }
}
