<?php

namespace App\Filament\SuperAdmin\Resources\CourseResource\RelationManagers;

use App\Exports\CourseAttendancesExport;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use pxlrbt\FilamentExcel\Actions\Tables\ExportAction;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;

class AttendancesRelationManager extends RelationManager
{
    protected static string $relationship = 'attendances';

    protected static ?string $recordTitleAttribute = 'id';

    public static function getTitle(\Illuminate\Database\Eloquent\Model $ownerRecord, string $pageClass): string
    {
        return __('admin.relation.attendances');
    }

    public static function getPluralRecordLabel(): ?string
    {
        return __('admin.relation.attendances');
    }

    /**
     * @deprecated Use `getModelLabel()` instead.
     */
    public static function getRecordLabel(): ?string
    {
        return __('admin.relation.attendance');
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('course_trainee_id')
                    ->relationship('courseTrainee', 'id')
                    ->getOptionLabelFromRecordUsing(function ($record) {
                        return $record->trainee->name;
                    })
                    ->required()
                    ->preload()
                    ->searchable()
                    ->label(__('attendance.trainee')),
                Forms\Components\DateTimePicker::make('attended_at')
                    ->nullable()
                    ->label(__('attendance.attended_at')),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('attended_at')
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->sortable()
                    ->searchable()
                    ->label('ID'),

                Tables\Columns\TextColumn::make('trainee.name')
                    ->label(__('attendance.trainee'))
                    ->searchable()
                    ->sortable(),

                Tables\Columns\TextColumn::make('attended_at')
                    ->dateTime()
                    ->label(__('attendance.attended_at')),

                Tables\Columns\ToggleColumn::make('is_attended')
                    ->label(__('attendance.is_present'))
                    ->sortable(),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make(),
                Tables\Actions\Action::make('register_all_as_not_attended')
                    ->label(__('attendance.register_all_as_not_attended'))
                    ->action(function () {
                        $this->ownerRecord->attendances()->update(['is_attended' => false]);
                    })
                    ->requiresConfirmation(),

                Tables\Actions\Action::make('register_all_as_attended')
                    ->label(__('attendance.register_all_as_attended'))
                    ->action(function () {
                        $course = $this->ownerRecord;
                        $days = $course->getDays();
                        $course->attendances()->delete();
                        foreach ($course->trainees as $trainee) {
                            foreach ($days as $day) {
                                $course->attendances()->create([
                                    'course_trainee_id' => $trainee->pivot->id,
                                    'attended_at' => $day,
                                ]);
                            }
                        }
                    })
                    ->requiresConfirmation(),

                ExportAction::make()->exports([
                    CourseAttendancesExport::make()
                        ->askForFilename(
                            default: $this->ownerRecord->title.' - Attendances - '.date('Y-m-d').' - export',
                            label : __('action.export_file_name')
                        ),
                ]),

            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                //                Tables\Actions\Action::make('register_as_not_attended')
                //                    ->label(__('attendance.register_as_not_attended'))
                //                    ->visible(function ($record) {
                //                        return $record->attended_at;
                //                    })
                //                    ->action(function ($record) {
                //                        $record->update(['is_attended' => false]);
                //                    })
                //                    ->requiresConfirmation(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\BulkAction::make('register_as_not_attended')
                        ->label(__('attendance.register_as_not_attended'))
                        ->action(function ($records) {
                            foreach ($records as $record) {
                                $record->update(['is_attended' => false]);
                            }
                        })
                        ->requiresConfirmation(),

                    ExportBulkAction::make()->exports([
                        CourseAttendancesExport::make()
                            ->askForFilename(
                                default: $this->ownerRecord->title.' - Attendances - '.date('Y-m-d').' - export',
                                label : __('action.export_file_name')
                            ),
                    ]),
                ]),
            ])
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }
}
