<?php

namespace App\Filament\GeneralUser\Pages;

use Filament\Forms\Components\Component;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Illuminate\Contracts\Support\Htmlable;
use Illuminate\Support\HtmlString;

class IdCardPage extends Page implements HasForms
{
    use InteractsWithForms;

    public $name;

    public $national_id;

    public $work;

    public $employer;

    public $photo;

    public $confirm_modal;

    public bool $isConfirmed;

    protected static ?string $navigationIcon = 'heroicon-o-user';

    protected static string $view = 'filament.general-user.pages.id-card-page';

    public static function getNavigationLabel(): string
    {
        return __('profile.identity');
    }

    public function getHeading(): string|Htmlable
    {
        return __('profile.identity');
    }

    public function getTitle(): string
    {
        return __('profile.identity');
    }

    public function getCancelButtonUrlProperty()
    {
        return static::getUrl();
    }

    public function getBreadcrumbs(): array
    {
        return [
            url()->current() => __('profile.title'),
        ];
    }

    public function mount()
    {
        $user = auth()->user();
        $idCard = $user->idCard;
        $this->form->fill([
            'name' => $idCard->name ?? $user->name,
            'national_id' => $idCard->national_id ?? $user->national_id,
            'work' => $idCard->work ?? '',
            'employer' => $idCard->employer ?? '',
            'photo' => $idCard->photo ?? '',
        ]);

        $this->isConfirmed = $idCard?->isConfirmed() ?? false;
    }

    public function notify($type, $msg)
    {
        Notification::make()
            ->body($msg)
            ->$type()
            ->send();
    }

    public function showId()
    {
        $this->redirect(route('user.id_card'));
    }

    public function submit()
    {
        $this->form->getState();

        $user = auth()->user();

        // get first element form photo array , get value
        $photo = array_values($this->photo)[0];

        $user->idCard()->updateOrCreate(
            ['general_user_id' => $user->id],
            [
                'name' => $this->name,
                'national_id' => $this->national_id,
                'work' => $this->work,
                'employer' => $this->employer,
                'photo' => $photo,
            ]
        );

        $this->notify('success', __('profile.id_card_updated'));
    }

    public function confirm()
    {
        if ($this->isConfirmed) {
            $this->notify('error', __('profile.id_already_confirmed'));

            return;
        }

        auth()->user()->idCard()->update(['confirmed_at' => now()]);

        $this->notify('success', __('profile.id_card_confirmed'));
    }

    protected function getUpdateNationaIdFormComponent(): Component
    {
        return TextInput::make('national_id')
            ->required()
            ->numeric()
            ->autofocus()
            ->unique(ignoreRecord: true)
            ->disabled()
            ->label(__('input.national_id'))
            ->placeholder(__('input.national_id_placeholder'))
            ->helperText(__('input.national_id_helper_text'));
    }

    protected function getNameFormComponent(): Component
    {
        return TextInput::make('name')
            ->label(__('filament-panels::pages/auth/edit-profile.form.name.label'))
            ->required()
            ->maxLength(255)
            ->default(auth()->user()->name)
            ->autofocus();
    }

    protected function getPhotoFormComponent(): Component
    {
        return FileUpload::make('photo')
            ->label(__('input.personal_image'))
            ->required()
            ->previewable()
            ->downloadable()
            ->image()
            ->imageEditor()
            ->imageEditorAspectRatios([
                '1:1',
            ])
            ->helperText(new HtmlString(' - تكون ملونة ذات خلفية بيضاء.
                    <br>
                    - أن تكون الصورة حديثة تم التقاطها قبل 4 أشهر كحد أقصى.
                    <br>
                    - تكون ملامح الوجه واضحة وبدون تعابير.
                    <br>
                    - تكون الصوره بوجه متجهه للكاميرا بشكل مستقيم.'))
            ->autofocus();
    }

    protected function getWorkFormComponent(): Component
    {
        return Select::make('work')
            ->label(__('input.work'))
            ->options([
                'manager' => __('input.manager'),
                'moderator' => __('input.moderator'),
            ])
            ->required()
            ->autofocus();
    }

    protected function getEmployerFormComponent(): Component
    {
        return Select::make('employer')
            ->label(__('input.employer'))
            ->options(config('edu.employers'))
            ->required()
            ->autofocus();
    }

    public function form(Form $form): Form
    {
        $idCard = auth()->user()->idCard;
        $isConfirmed = $idCard?->isConfirmed() ?? false;

        return $form
            ->schema([
                Section::make()
                    ->schema([
                        $this->getPhotoFormComponent()
                            ->disabled($isConfirmed),
                    ]),

                Section::make()
                    ->columns(2)
                    ->schema([
                        $this->getUpdateNationaIdFormComponent(),
                        $this->getNameFormComponent()
                            ->disabled($isConfirmed),
                    ]),

                Section::make()
                    ->columns(2)
                    ->schema([
                        $this->getWorkFormComponent()
                            ->disabled($isConfirmed),
                        $this->getEmployerFormComponent()
                            ->disabled($isConfirmed),
                    ]),
            ])
            ->context(static::class);
    }
}
