<?php

namespace App\Filament\Employee\Resources\Trainer;

use App\Filament\Employee\Resources\Trainer\MyCourseResource\Pages;
use App\Filament\Employee\Resources\Trainer\MyCourseResource\RelationManagers\AttendancesRelationManager;
use App\Filament\Employee\Resources\Trainer\MyCourseResource\RelationManagers\TraineesRelationManager;
use App\Models\Course;
use Filament\Forms\Components\DatePicker;
use Filament\Infolists\Components\IconEntry;
use Filament\Infolists\Components\RepeatableEntry;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Infolist;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\Summarizers\Average;
use Filament\Tables\Columns\Summarizers\Count;
use Filament\Tables\Columns\Summarizers\Range;
use Filament\Tables\Columns\Summarizers\Sum;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class MyCourseResource extends Resource
{
    protected static ?string $model = Course::class;

    protected static ?string $navigationIcon = 'heroicon-o-presentation-chart-line';

    protected static ?string $recordTitleAttribute = 'title';

    public static function canViewAny(): bool
    {
        return auth()->user()->is_trainer;
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->OfCurrentTrainer();
    }

    public static function getNavigationBadge(): ?string
    {
        return static::$model::OfCurrentTrainer()->count();
    }

    public static function getNavigationLabel(): string
    {
        return __('admin.my_courses');
    }

    public static function getNavigationGroup(): ?string
    {
        return __('admin.group.trainer');
    }

    /**
     * @deprecated Use `getModelLabel()` instead.
     */
    public static function getPluralLabel(): ?string
    {
        return __('admin.my_courses');
    }

    /**
     * @deprecated Use `getModelLabel()` instead.
     */
    public static function getLabel(): ?string
    {
        return __('admin.my_course');
    }

    public static function infolist(Infolist $infolist): Infolist
    {
        return $infolist->schema([
            Section::make()
                ->columns(2)
                ->icon('heroicon-o-clock')
                ->heading(function ($record) {
                    return $record->title;
                })
                ->schema([
                    TextEntry::make('start_at')
                        ->label(__('input.start_at'))
                        ->columns(6)
                        ->dateTime(),
                    TextEntry::make('end_at')
                        ->label(__('input.end_at'))
                        ->columns(6)
                        ->dateTime(),
                ]),

            Section::make('course_detail')
                ->columns(4)
                ->icon('heroicon-o-academic-cap')
                ->heading(null)
                ->schema([
                    TextEntry::make('type.name')
                        ->visible(fn ($record) => $record->course_type_id)
                        ->label(__('input.type')),

                    TextEntry::make('field.name')
                        ->visible(fn ($record) => $record->course_field_id)
                        ->label(__('input.field')),

                    IconEntry::make('is_online')
                        ->label(__('input.is_online'))
                        ->boolean(),

                    IconEntry::make('is_for_all_offices')
                        ->label(__('input.is_for_all_offices'))
                        ->boolean(),

                    IconEntry::make('is_for_boys')
                        ->boolean()
                        ->label(__('input.is_for_boys')),

                    IconEntry::make('is_for_girls')
                        ->boolean()
                        ->label(__('input.is_for_girls')),

                    IconEntry::make('is_for_edu_job')
                        ->boolean()
                        ->label(__('input.is_for_edu_job')),

                    IconEntry::make('is_for_manager_job')
                        ->boolean()
                        ->label(__('input.is_for_manager_job')),

                    Section::make()
                        ->columns(2)
                        ->visible(fn ($record) => $record->is_online)
                        ->icon('heroicon-o-link')
                        ->schema([
                            TextEntry::make('meeting_link')
                                ->copyable()
                                ->label(__('input.meeting_link'))
                                ->hint(__('input.meeting_link_hint')),
                            TextEntry::make('meeting_password')
                                ->copyable()
                                ->label(__('input.meeting_password'))
                                ->hint(__('input.meeting_password_hint')),
                        ]),
                ]),

            Section::make()
                ->icon('heroicon-o-building-office-2')
                ->visible(fn ($record) => $record->offices->count())
                ->schema([
                    RepeatableEntry::make('offices')
                        ->label(__('admin.offices'))
                        ->columnSpanFull()
                        ->schema([
                            TextEntry::make('name')
                                ->label(''),
                        ])
                        ->grid(4),
                ]),

            Section::make()
                ->icon('heroicon-o-user-group')
                ->visible(fn ($record) => $record->trainers->count())
                ->schema([
                    RepeatableEntry::make('trainers')
                        ->label(__('admin.the_trainers'))
                        ->columnSpanFull()
                        ->schema([
                            TextEntry::make('name')
                                ->label(''),
                        ])
                        ->grid(4),
                ]),
        ])
            ->columns(2);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('type.name')
                    ->sortable()
                    ->searchable()
                    ->toggleable()
                    ->label(__('input.type')),

                Tables\Columns\TextColumn::make('field.name')
                    ->sortable()
                    ->searchable()
                    ->toggleable()
                    ->label(__('input.field')),

                Tables\Columns\TextColumn::make('title')
                    ->sortable()
                    ->searchable()
                    ->label(__('input.title')),

                Tables\Columns\TextColumn::make('start_at')
                    ->date()
                    ->sortable()
                    ->toggleable()
                    ->searchable()
                    ->label(__('input.start_at')),
                Tables\Columns\TextColumn::make('end_at')
                    ->date()
                    ->sortable()
                    ->toggleable()
                    ->searchable()
                    ->label(__('input.end_at')),

                Tables\Columns\TextColumn::make('hours')
                    ->sortable()
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->summarize([
                        Average::make(),
                        Range::make(),
                        Count::make(),
                        Sum::make(),
                    ])
                    ->label(__('input.hours')),

                Tables\Columns\TextColumn::make('trainers_count')
                    ->counts('trainers')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->summarize([
                        Average::make(),
                        Range::make(),
                        Count::make(),
                        Sum::make(),
                    ])
                    ->label(__('input.trainers_count')),

                Tables\Columns\TextColumn::make('trainees_count')
                    ->counts('trainees')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->summarize([
                        Average::make(),
                        Range::make(),
                        Count::make(),
                        Sum::make(),
                    ])
                    ->label(__('input.trainees_count')),

                Tables\Columns\TextColumn::make('offices_count')
                    ->counts('offices')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->summarize([
                        Average::make(),
                        Range::make(),
                        Count::make(),
                        Sum::make(),
                    ])
                    ->label(__('input.offices_count')),

                Tables\Columns\TextColumn::make('max_trainees')
                    ->sortable()
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->summarize([
                        Average::make(),
                        Range::make(),
                        Count::make(),
                        Sum::make(),
                    ])
                    ->label(__('input.max_trainees')),
                Tables\Columns\TextColumn::make('max_trainees_per_school')
                    ->sortable()
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->summarize([
                        Average::make(),
                        Range::make(),
                        Count::make(),
                        Sum::make(),
                    ])
                    ->label(__('input.max_trainees_per_school')),

                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->label(__('input.created_at'))
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->label(__('input.updated_at'))
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Filter::make('is_started')
                    ->label(__('input.is_started'))
                    ->toggle()
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->Started();
                    }),

                Filter::make('is_not_started')
                    ->label(__('input.is_not_started'))
                    ->toggle()
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->NotStarted();
                    }),

                Filter::make('is_ended')
                    ->label(__('input.is_ended'))
                    ->toggle()
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->Ended();
                    }),

                Filter::make('is_not_ended')
                    ->label(__('input.is_not_ended'))
                    ->toggle()
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->NotEnded();
                    }),

                Filter::make('between_start_end')
                    ->form([
                        DatePicker::make('start_at')
                            ->label(__('input.from')),
                        DatePicker::make('end_at')
                            ->label(__('input.to')),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['start_at'],
                                fn (Builder $query, $date): Builder => $query->whereDate('start_at', '>=', $date),
                            )
                            ->when(
                                $data['end_at'],
                                fn (Builder $query, $date): Builder => $query->whereDate('end_at', '<=', $date),
                            );
                    })
                    ->label(__('input.between_start_end')),

                TernaryFilter::make('is_online')
                    ->label(__('input.is_online')),

                SelectFilter::make('type')
                    ->label(__('input.type'))
                    ->searchable()
                    ->preload()
                    ->relationship('type', 'name')
                    ->multiple(),

                SelectFilter::make('field')
                    ->label(__('input.field'))
                    ->searchable()
                    ->preload()
                    ->relationship('field', 'name')
                    ->multiple(),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            TraineesRelationManager::class,
            AttendancesRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListMyCourses::route('/'),
            'view' => Pages\ViewMyCourse::route('/{record}'),
        ];
    }
}
