<?php

namespace App\Filament\Employee\Resources\Trainee;

use App\Filament\Employee\Resources\Trainee\MyCourseResource\Pages;
use App\Models\Course;
use Filament\Forms\Components\DatePicker;
use Filament\Infolists\Components\IconEntry;
use Filament\Infolists\Components\RepeatableEntry;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Infolist;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Support\Enums\ActionSize;
use Filament\Tables;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\Layout\Panel;
use Filament\Tables\Columns\Layout\Split;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class MyCourseResource extends Resource
{
    protected static ?string $model = Course::class;

    protected static ?string $navigationIcon = 'heroicon-o-squares-plus';

    protected static ?string $recordTitleAttribute = 'title';

    public static function canViewAny(): bool
    {
        return auth()->user()->is_trainee;
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->OfCurrentTrainee();
    }

    public static function getNavigationBadge(): ?string
    {
        return static::$model::OfCurrentTrainee()->count();
    }

    public static function getNavigationLabel(): string
    {
        return __('admin.my_courses');
    }

    public static function getNavigationGroup(): ?string
    {
        return __('admin.group.trainee');
    }

    /**
     * @deprecated Use `getModelLabel()` instead.
     */
    public static function getPluralLabel(): ?string
    {
        return __('admin.my_courses');
    }

    /**
     * @deprecated Use `getModelLabel()` instead.
     */
    public static function getLabel(): ?string
    {
        return __('admin.my_course');
    }

    public static function infolist(Infolist $infolist): Infolist
    {
        return $infolist->schema([
            Section::make()
                ->columns(2)
                ->icon('heroicon-o-clock')
                ->heading(function ($record) {
                    return $record->title;
                })
                ->schema([
                    TextEntry::make('start_at')
                        ->label(__('input.start_at'))
                        ->columns(6)
                        ->dateTime(),
                    TextEntry::make('end_at')
                        ->label(__('input.end_at'))
                        ->columns(6)
                        ->dateTime(),
                ]),

            Section::make('course_detail')
                ->columns(4)
                ->icon('heroicon-o-academic-cap')
                ->heading(null)
                ->schema([
                    TextEntry::make('type.name')
                        ->visible(fn ($record) => $record->course_type_id)
                        ->label(__('input.type')),

                    TextEntry::make('field.name')
                        ->visible(fn ($record) => $record->course_field_id)
                        ->label(__('input.field')),

                    IconEntry::make('is_online')
                        ->label(__('input.is_online'))
                        ->boolean(),

                    IconEntry::make('is_for_all_offices')
                        ->label(__('input.is_for_all_offices'))
                        ->boolean(),

                    Section::make()
                        ->columns(2)
                        ->visible(fn ($record) => $record->is_online)
                        ->icon('heroicon-o-link')
                        ->schema([
                            TextEntry::make('meeting_link')
                                ->copyable()
                                ->label(__('input.meeting_link'))
                                ->hint(__('input.meeting_link_hint')),
                            TextEntry::make('meeting_password')
                                ->copyable()
                                ->label(__('input.meeting_password'))
                                ->hint(__('input.meeting_password_hint')),
                        ]),
                ]),

            Section::make()
                ->icon('heroicon-o-building-office-2')
                ->visible(fn ($record) => $record->offices->count())
                ->schema([
                    RepeatableEntry::make('offices')
                        ->label(__('admin.offices'))
                        ->columnSpanFull()
                        ->schema([
                            TextEntry::make('name')
                                ->label(''),
                        ])
                        ->grid(4),
                ]),

            Section::make()
                ->icon('heroicon-o-user-group')
                ->visible(fn ($record) => $record->trainers->count())
                ->schema([
                    RepeatableEntry::make('trainers')
                        ->label(__('admin.the_trainers'))
                        ->columnSpanFull()
                        ->schema([
                            TextEntry::make('name')
                                ->label(''),
                        ])
                        ->grid(4),
                ]),
        ])
            ->columns(2);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->contentGrid([
                'xl' => 2,
            ])
            ->paginated([6, 8])
            ->recordClasses(fn (Course $record) => match ($record->timingStatus) {
                'ended' => 'bg-red-100 text-red-800',
                'tomorrow' => 'bg-yellow-100 text-yellow-800',
                'started' => 'bg-blue-100 text-blue-800',
                'not_started' => 'bg-gray-100 text-gray-800',
                'today' => 'bg-green-100 text-green-800',
            })
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->sortable()
                    ->searchable()
                    ->extraAttributes(['class' => 'mb-4'])
                    ->icon('heroicon-o-academic-cap')
                    ->label(__('input.title')),

                Split::make([
                    Tables\Columns\TextColumn::make('start_at')
                        ->date()
                        ->sortable()
                        ->searchable()
                        ->tooltip(function ($state) {
                            return $state->diffForHumans();
                        })
                        ->label(__('input.start_at'))
                        ->icon('heroicon-o-arrow-left')
                        ->formatStateUsing(function ($state) {
                            return $state->format('d-m-Y h:iA');
                        }),
                    Tables\Columns\TextColumn::make('end_at')
                        ->date()
                        ->sortable()
                        ->searchable()
                        ->label(__('input.end_at'))
                        ->tooltip(function ($state) {
                            return $state->diffForHumans();
                        })
                        ->icon('heroicon-o-arrow-left')
                        ->formatStateUsing(function ($state) {
                            return $state->format('d-m-Y h:iA');
                        }),
                ]),

                Panel::make([
                    Split::make([
                        Tables\Columns\TextColumn::make('type.name')
                            ->sortable()
                            ->searchable()
                            ->icon('heroicon-o-tag')
                            ->label(__('input.type')),

                        Tables\Columns\TextColumn::make('field.name')
                            ->sortable()
                            ->searchable()
                            ->icon('heroicon-o-hashtag')
                            ->label(__('input.field')),

                        Tables\Columns\TextColumn::make('hours')
                            ->sortable()
                            ->searchable()
                            ->icon('heroicon-o-clock')
                            ->formatStateUsing(function ($state) {
                                if ($state == 1) {
                                    return __('hour.hour');
                                } elseif ($state == 2) {
                                    return __('hour.two_hours');
                                } elseif ($state > 2 && $state < 10) {
                                    return $state.' '.__('hour.hours');
                                } else {
                                    return $state.' '.__('hour.hour');
                                }
                            })
                            ->label(__('input.hours')),
                    ])->from('md'),
                ])->collapsed(false),
            ])
            ->filters([
                Filter::make('is_started')
                    ->label(__('input.is_started'))
                    ->toggle()
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->Started();
                    }),

                Filter::make('is_not_started')
                    ->label(__('input.is_not_started'))
                    ->toggle()
                    ->default(true)
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->NotStarted();
                    }),

                Filter::make('is_ended')
                    ->label(__('input.is_ended'))
                    ->toggle()
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->Ended();
                    }),

                Filter::make('is_not_ended')
                    ->label(__('input.is_not_ended'))
                    ->toggle()
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->NotEnded();
                    }),

                Filter::make('between_start_end')
                    ->form([
                        DatePicker::make('start_at')
                            ->label(__('input.from')),
                        DatePicker::make('end_at')
                            ->label(__('input.to')),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['start_at'],
                                fn (Builder $query, $date): Builder => $query->whereDate('start_at', '>=', $date),
                            )
                            ->when(
                                $data['end_at'],
                                fn (Builder $query, $date): Builder => $query->whereDate('end_at', '<=', $date),
                            );
                    })
                    ->label(__('input.between_start_end')),

                TernaryFilter::make('is_online')
                    ->label(__('input.is_online')),

                SelectFilter::make('type')
                    ->label(__('input.type'))
                    ->searchable()
                    ->preload()
                    ->relationship('type', 'name')
                    ->multiple(),

                SelectFilter::make('field')
                    ->label(__('input.field'))
                    ->searchable()
                    ->preload()
                    ->relationship('field', 'name')
                    ->multiple(),
            ], )
            ->filtersFormColumns(4)
            ->persistFiltersInSession()
            ->filtersTriggerAction(
                fn (Action $action) => $action
                    ->button()
                    ->label(__('admin.filter')),
            )
            ->actions([
                Tables\Actions\ViewAction::make(),
                self::leaveAction(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ManageMyCourses::route('/'),
        ];
    }

    private static function leaveAction(): Action
    {
        return Action::make('leave_course')
            ->label(__('course.leave_btn'))
            ->size(ActionSize::Large)
            ->icon('heroicon-m-minus')
            ->visible(function (Course $course) {
                return auth()->user()->traineeCourses()->where('course_id', $course->id)->exists();
            })
            ->disabled(function (Course $course) {
                return ! $course->stillCanJoinOrLeave();
            })
            ->tooltip(function (Course $course) {
                $last_time_to_leave = $course->last_time_to_join;

                if ($last_time_to_leave->isPast()) {
                    return __('course.ended_lave_btn_tooltip');
                }

                return __('course.leave_btn_tooltip', ['date' => $last_time_to_leave->format('d/m/Y H:i')]);
            })
            ->outlined()
            ->color('danger')
            ->action(function (Course $course) {
                $last_time_to_leave = $course->last_time_to_join;

                if ($last_time_to_leave->isPast()) {
                    Notification::make()
                        ->title(__('course.ended_lave_btn_tooltip'))
                        ->danger()
                        ->body(__('course.ended_lave_btn_tooltip'))
                        ->send();

                    return;
                }

                $user = auth()->user();

                if (! $user->traineeCourses()->where('course_id', $course->id)->exists()) {
                    Notification::make()
                        ->title(__('course.not_joined'))
                        ->danger()
                        ->body(__('course.not_joined'))
                        ->send();

                    return;
                }

                $user->traineeCourses()->detach($course->id);

                Notification::make()
                    ->title(__('course.left'))
                    ->success()
                    ->body(__('course.left'))
                    ->send();
            })
            ->requiresConfirmation();
    }
}
