<?php

namespace App\Filament\Admin\Resources;

use App\Enums\Gender;
use App\Filament\Admin\Resources\UserResource\Pages;
use App\Models\User;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class UserResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function getNavigationBadge(): ?string
    {
        return static::$model::AdminControl()->count();
    }

    public static function getNavigationLabel(): string
    {
        return __('admin.users');
    }

    public static function getNavigationGroup(): ?string
    {
        return __('admin.group.users');
    }

    /**
     * @deprecated Use `getModelLabel()` instead.
     */
    public static function getPluralLabel(): ?string
    {
        return __('admin.users');
    }

    /**
     * @deprecated Use `getModelLabel()` instead.
     */
    public static function getLabel(): ?string
    {
        return __('admin.user');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('national_id')
                    ->required()
                    ->numeric()
                    ->autofocus()
                    ->disabledOn('edit')
                    ->unique(ignoreRecord: true)
                    ->label(__('input.national_id'))
                    ->placeholder(__('input.national_id_placeholder'))
                    ->helperText(__('input.national_id_helper_text')),

                TextInput::make('name')
                    ->required()
                    ->autofocus()
                    ->label(__('input.name'))
                    ->placeholder(__('input.name_placeholder'))
                    ->helperText(__('input.name_helper_text')),

                Select::make('office_id')
                    ->label(__('input.office'))
                    ->relationship('office', 'name')
                    ->searchable()
                    ->reactive()
                    ->afterStateUpdated(function ($set) {
                        $set('school_id', null);
                    })
                    ->placeholder(__('input.office_placeholder'))
                    ->helperText(__('input.office_helper_text')),

                Select::make('school_id')
                    ->label(__('input.school'))
                    ->relationship('school', 'name', function ($query, $get) {
                        if ($get('office_id')) {
                            $query->where('office_id', $get('office_id'));
                        } else {
                            $query->where('office_id', 0);
                        }

                        return $query;
                    })
                    ->searchable()
                    ->placeholder(__('input.school_placeholder'))
                    ->helperText(__('input.school_helper_text')),

                TextInput::make('email')
                    ->required()
                    ->email()
                    ->unique(ignoreRecord: true)
                    ->label(__('input.email'))
                    ->placeholder(__('input.email_placeholder'))
                    ->helperText(__('input.email_helper_text')),

                TextInput::make('phone')
                    ->tel()
                    ->unique(ignoreRecord: true)
                    ->label(__('input.phone'))
                    ->placeholder(__('input.phone_placeholder'))
                    ->helperText(__('input.phone_helper_text')),

                TextInput::make('password')
                    ->nullable()
                    ->password()
                    ->confirmed()
                    ->label(__('input.password'))
                    ->placeholder(__('input.password_placeholder'))
                    ->helperText(__('input.password_helper_text')),

                TextInput::make('password_confirmation')
                    ->nullable()
                    ->password()
                    ->label(__('input.password_confirmation'))
                    ->placeholder(__('input.password_confirmation_placeholder'))
                    ->helperText(__('input.password_confirmation_helper_text')),

                Select::make('gender')
                    ->columnSpanFull()
                    ->disabledOn('edit')
                    ->options(Gender::valuesForSelect())
                    ->label(__('input.gender')),

                Section::make()
                    ->columns([
                        '2xl' => 4,
                    ])
                    ->schema([
                        Toggle::make('is_trainer')
                            ->disabledOn('edit')
                            ->label(__('input.is_trainer')),

                        Toggle::make('is_trainee')
                            ->disabledOn('edit')
                            ->label(__('input.is_trainee')),

                        Toggle::make('has_edu_job')
                            ->disabledOn('edit')
                            ->label(__('input.has_edu_job')),

                        Toggle::make('has_manager_job')
                            ->disabledOn('edit')
                            ->label(__('input.has_manager_job')),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->sortable()
                    ->searchable()
                    ->label('ID'),
                Tables\Columns\TextColumn::make('national_id')
                    ->sortable()
                    ->searchable()
                    ->label(__('input.national_id')),

                Tables\Columns\TextColumn::make('name')
                    ->sortable()
                    ->searchable()
                    ->label(__('input.name')),

                Tables\Columns\TextColumn::make('email')
                    ->sortable()
                    ->searchable()
                    ->label(__('input.email')),

                Tables\Columns\TextColumn::make('phone')
                    ->sortable()
                    ->searchable()
                    ->toggleable()
                    ->label(__('input.phone')),

                Tables\Columns\TextColumn::make('office.name')
                    ->sortable()
                    ->searchable()
                    ->toggleable()
                    ->label(__('input.office')),

                Tables\Columns\TextColumn::make('school.name')
                    ->sortable()
                    ->searchable()
                    ->toggleable()
                    ->label(__('input.school')),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ])
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            // 'create' => Pages\CreateUser::route('/create'),
            'view' => Pages\ViewUser::route('/{record}'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ])
            ->AdminControl();
    }
}
