<?php

namespace App\Filament\Admin\Resources;

use App\Filament\Admin\Resources\CourseResource\Pages;
use App\Filament\Admin\Resources\CourseResource\RelationManagers\AttendancesRelationManager;
use App\Filament\Admin\Resources\CourseResource\RelationManagers\OfficesRelationManager;
use App\Filament\Admin\Resources\CourseResource\RelationManagers\TraineesRelationManager;
use App\Filament\Admin\Resources\CourseResource\RelationManagers\TrainersRelationManager;
use App\Models\Course;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\Summarizers\Average;
use Filament\Tables\Columns\Summarizers\Count;
use Filament\Tables\Columns\Summarizers\Range;
use Filament\Tables\Columns\Summarizers\Sum;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Carbon;

class CourseResource extends Resource
{
    protected static ?string $model = Course::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function getNavigationBadge(): ?string
    {
        return static::$model::count();
    }

    public static function getNavigationLabel(): string
    {
        return __('admin.courses');
    }

    public static function getNavigationGroup(): ?string
    {
        return __('admin.group.courses');
    }

    /**
     * @deprecated Use `getModelLabel()` instead.
     */
    public static function getPluralLabel(): ?string
    {
        return __('admin.courses');
    }

    /**
     * @deprecated Use `getModelLabel()` instead.
     */
    public static function getLabel(): ?string
    {
        return __('admin.course');
    }

    public static function form(Form $form): Form
    {
        $admin = auth('admin')->user();

        return $form
            ->schema([
                Section::make()
                    ->columns([
                        '2xl' => 2,
                    ])
                    ->schema([
                        Select::make('course_type_id')
                            ->relationship('type', 'name')
                            ->required()
                            ->label(__('input.type')),

                        Select::make('course_field_id')
                            ->relationship('field', 'name')
                            ->required()
                            ->label(__('input.field')),

                        TextInput::make('title')
                            ->label(__('input.title'))
                            ->required()
                            ->columnSpanFull()
                            ->maxLength(255),
                    ]),
                Section::make()
                    ->columns([
                        '2xl' => 2,
                    ])
                    ->schema([
                        DateTimePicker::make('start_at')
                            ->label(__('input.start_at'))
                            ->columns(6)
                            ->required(),
                        DateTimePicker::make('end_at')
                            ->label(__('input.end_at'))
                            ->columns(6)
                            ->required(),
                    ]),

                Section::make()
                    ->columns([
                        '2xl' => 3,
                    ])
                    ->schema([
                        TextInput::make('hours')
                            ->numeric()
                            ->label(__('input.hours'))
                            ->required(),
                        TextInput::make('max_trainees')
                            ->numeric()
                            ->label(__('input.max_trainees'))
                            ->required(),
                        TextInput::make('max_trainees_per_school')
                            ->numeric()
                            ->label(__('input.max_trainees_per_school'))
                            ->required(),
                    ]),

                Section::make()
                    ->columns([
                        '2xl' => 2,
                    ])
                    ->schema([
                        Toggle::make('is_online')
                            ->label(__('input.is_online')),

                        Toggle::make('is_for_all_offices')
                            ->default(true)
                            ->label(__('input.is_for_all_offices')),

                        Toggle::make('is_for_boys')
                            ->default($admin->is_has_males)
                            ->disabled($admin->is_has_males && ! $admin->is_has_females)
                            ->visible($admin->is_has_males)
                            ->label(__('input.is_for_boys')),

                        Toggle::make('is_for_girls')
                            ->default($admin->is_has_females)
                            ->disabled($admin->is_has_females && ! $admin->is_has_males)
                            ->visible($admin->is_has_females)
                            ->label(__('input.is_for_girls')),

                        Toggle::make('is_for_edu_job')
                            ->default($admin->is_has_edu_job)
                            ->visible($admin->is_has_edu_job)
                            ->label(__('input.is_for_edu_job')),

                        Toggle::make('is_for_manager_job')
                            ->visible($admin->is_has_manager_job)
                            ->label(__('input.is_for_manager_job')),

                    ]),

                Section::make()
                    ->columns([
                        '2xl' => 3,
                    ])
                    ->visible(function ($get) {
                        return $get('is_online');
                    })
                    ->schema([
                        TextInput::make('meeting_link')
                            ->visible(function ($get) {
                                return $get('is_online');
                            })->required(function ($get) {
                                return $get('is_online');
                            })
                            ->label(__('input.meeting_link'))
                            ->hint(__('input.meeting_link_hint'))
                            ->maxLength(255),
                        TextInput::make('meeting_password')
                            ->visible(function ($get) {
                                return $get('is_online');
                            })
                            ->label(__('input.meeting_password'))
                            ->hint(__('input.meeting_password_hint'))
                            ->maxLength(255),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('type.name')
                    ->sortable()
                    ->searchable()
                    ->toggleable()
                    ->label(__('input.type')),

                Tables\Columns\TextColumn::make('field.name')
                    ->sortable()
                    ->searchable()
                    ->toggleable()
                    ->label(__('input.field')),

                Tables\Columns\TextColumn::make('title')
                    ->sortable()
                    ->searchable()
                    ->label(__('input.title')),

                Tables\Columns\TextColumn::make('start_at')
                    ->date()
                    ->sortable()
                    ->toggleable()
                    ->searchable()
                    ->label(__('input.start_at')),
                Tables\Columns\TextColumn::make('end_at')
                    ->date()
                    ->sortable()
                    ->toggleable()
                    ->searchable()
                    ->label(__('input.end_at')),

                Tables\Columns\TextColumn::make('hours')
                    ->sortable()
                    ->searchable()
                    ->summarize([
                        Average::make(),
                        Range::make(),
                        Count::make(),
                        Sum::make(),
                    ])
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->label(__('input.hours')),

                Tables\Columns\TextColumn::make('trainers_count')
                    ->counts('trainers')
                    ->sortable()
                    ->summarize([
                        Average::make(),
                        Range::make(),
                        Count::make(),
                        Sum::make(),
                    ])
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->label(__('input.trainers_count')),

                Tables\Columns\TextColumn::make('trainees_count')
                    ->counts('trainees')
                    ->sortable()
                    ->summarize([
                        Average::make(),
                        Range::make(),
                        Count::make(),
                        Sum::make(),
                    ])
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->label(__('input.trainees_count')),

                Tables\Columns\TextColumn::make('offices_count')
                    ->counts('offices')
                    ->sortable()
                    ->summarize([
                        Average::make(),
                        Range::make(),
                        Count::make(),
                        Sum::make(),
                    ])
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->label(__('input.offices_count')),

                Tables\Columns\TextColumn::make('max_trainees')
                    ->sortable()
                    ->searchable()
                    ->summarize([
                        Average::make(),
                        Range::make(),
                        Count::make(),
                        Sum::make(),
                    ])
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->label(__('input.max_trainees')),
                Tables\Columns\TextColumn::make('max_trainees_per_school')
                    ->sortable()
                    ->searchable()
                    ->summarize([
                        Average::make(),
                        Range::make(),
                        Count::make(),
                        Sum::make(),
                    ])
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->label(__('input.max_trainees_per_school')),

                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),

                SelectFilter::make('type')
                    ->label(__('input.type'))
                    ->searchable()
                    ->preload()
                    ->relationship('type', 'name')
                    ->multiple(),

                SelectFilter::make('field')
                    ->label(__('input.field'))
                    ->searchable()
                    ->preload()
                    ->relationship('field', 'name')
                    ->multiple(),

                self::addHoursRangeFilter(),
                self::addCreatedAtFilter(),
                self::addStartEndFilter(),

                SelectFilter::make('offices')
                    ->label(__('input.offices'))
                    ->searchable()
                    ->relationship('offices', 'name')
                    ->multiple(),

                TernaryFilter::make('is_online')
                    ->label(__('input.is_online')),

                TernaryFilter::make('is_for_all_offices')
                    ->label(__('input.is_for_all_offices')),

                TernaryFilter::make('is_for_boys')
                    ->label(__('input.is_for_boys')),

                TernaryFilter::make('is_for_girls')
                    ->label(__('input.is_for_girls')),

                TernaryFilter::make('is_for_edu_job')
                    ->label(__('input.is_for_edu_job')),

                TernaryFilter::make('is_for_manager_job')
                    ->label(__('input.is_for_manager_job')),

            ], layout: FiltersLayout::AboveContentCollapsible)
            ->filtersFormColumns(3)
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    //                    Tables\Actions\DeleteBulkAction::make(),
                    //                    Tables\Actions\ForceDeleteBulkAction::make(),
                    //                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ])
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            TrainersRelationManager::class,
            TraineesRelationManager::class,
            AttendancesRelationManager::class,
            OfficesRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListCourses::route('/'),
            'create' => Pages\CreateCourse::route('/create'),
            'view' => Pages\ViewCourse::route('/{record}'),
            'edit' => Pages\EditCourse::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }

    private static function addCreatedAtFilter(): Filter
    {
        return Filter::make('created_at')
            ->form([
                DatePicker::make('created_from')
                    ->label(__('input.created_from')),
                DatePicker::make('created_until')
                    ->label(__('input.created_until'))
                    ->default(now()),
            ])
            ->indicateUsing(function (array $data): array {
                $indicators = [];

                if ($data['created_from'] ?? null) {
                    $indicators['created_from'] = __('input.created_from').' '.Carbon::parse($data['created_from'])->toFormattedDateString();
                }

                if ($data['created_until'] ?? null) {
                    $indicators['created_until'] = __('input.created_until').' '.Carbon::parse($data['created_until'])->toFormattedDateString();
                }

                return $indicators;
            })
            ->query(function (Builder $query, array $data): Builder {
                return $query
                    ->when(
                        $data['created_from'],
                        fn (Builder $query, $date): Builder => $query->whereDate('created_at', '>=', $date),
                    )
                    ->when(
                        $data['created_until'],
                        fn (Builder $query, $date): Builder => $query->whereDate('created_at', '<=', $date),
                    );
            });
    }

    private static function addStartEndFilter(): Filter
    {
        return Filter::make('start_end')
            ->form([
                DatePicker::make('start_at')
                    ->label(__('input.start_at')),
                DatePicker::make('end_at')
                    ->label(__('input.end_at')),
            ])
            ->indicateUsing(function (array $data): array {
                $indicators = [];

                if ($data['start_at'] ?? null) {
                    $indicators['start_at'] = __('input.start_at').' '.Carbon::parse($data['start_at'])->toFormattedDateString();
                }

                if ($data['end_at'] ?? null) {
                    $indicators['end_at'] = __('input.end_at').' '.Carbon::parse($data['end_at'])->toFormattedDateString();
                }

                return $indicators;
            })
            ->query(function (Builder $query, array $data): Builder {
                return $query
                    ->when(
                        $data['start_at'],
                        fn (Builder $query, $date): Builder => $query->whereDate('start_at', '>=', $date),
                    )
                    ->when(
                        $data['end_at'],
                        fn (Builder $query, $date): Builder => $query->whereDate('end_at', '<=', $date),
                    );
            });
    }

    private static function addHoursRangeFilter(): Filter
    {
        $min = Course::min('hours');
        $max = Course::max('hours');

        return Filter::make('hours')
            ->form([
                TextInput::make('from')
                    ->numeric()
                    ->default($min)
                    ->minValue($min)
                    ->maxValue($max)
                    ->label(__('input.hours_from')),
                TextInput::make('to')
                    ->numeric()
                    ->default($max)
                    ->minValue($min)
                    ->maxValue($max)
                    ->label(__('input.hours_to')),
            ])
            ->indicateUsing(function (array $data): array {
                $indicators = [];

                if ($data['from'] ?? null) {
                    $indicators['from'] = __('input.hours_from').' '.$data['from'];
                }

                if ($data['to'] ?? null) {
                    $indicators['to'] = __('input.hours_to').' '.$data['to'];
                }

                return $indicators;
            })
            ->query(function (Builder $query, array $data): Builder {
                return $query
                    ->when(
                        $data['from'],
                        fn (Builder $query, $hours): Builder => $query->where('hours', '>=', $hours),
                    )
                    ->when(
                        $data['to'],
                        fn (Builder $query, $hours): Builder => $query->where('hours', '<=', $hours),
                    );
            });
    }
}
